<?php

namespace App\Http\Controllers;

use App\Models\Evaluation;
use App\Models\User;
use App\Models\Attendance;
use Illuminate\Http\Request;

class EvaluationController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }
    public function edit($id) {
        $evaluation = Evaluation::with('student')->findOrFail($id);
        return view('supervisors.evaluations.edit', compact('evaluation'));
    }


public function update(Request $request, $id)
{
    // Find the evaluation manually using the passed ID
    $evaluation = Evaluation::findOrFail($id);

    // Authorization check
    if ($evaluation->supervisor_id !== auth()->id()) {
        abort(403, 'Unauthorized');
    }

    $validated = $request->validate([
        'punctuality' => 'required|integer|min:1|max:10',
        'attitude' => 'required|integer|min:1|max:10',
        'skill_acquisition' => 'required|integer|min:1|max:10',
        'report_quality' => 'required|integer|min:1|max:10',
        'remarks' => 'nullable|string|max:1000',
    ]);

    $evaluation->update($validated);

    return redirect()->route('evaluations.index')->with('success', 'Evaluation updated successfully.');
}



public function index(Request $request)
{
    $supervisorId = auth()->id();
    $selectedStudentId = $request->student_id;

    $students = User::where('supervisor_id', $supervisorId)
                    ->where('role', 'student')
                    ->get();

    $evaluations = Evaluation::with('student')
                    ->where('supervisor_id', $supervisorId)
                    ->latest()
                    ->get();

    $averages = [
        'punctuality' => $evaluations->avg('punctuality'),
        'attitude' => $evaluations->avg('attitude'),
        'skill_acquisition' => $evaluations->avg('skill_acquisition'),
        'report_quality' => $evaluations->avg('report_quality'),
    ];

    $attendanceMap = \App\Models\Attendance::select('student_id')->distinct()->pluck('student_id')->flip()->toArray();

    $evaluationMap = \App\Models\Evaluation::where('supervisor_id', $supervisorId)
                        ->get()->keyBy('student_id')->toArray();

    // Safely fetch the selected student
    $selectedStudent = $selectedStudentId ? $students->where('id', $selectedStudentId)->first() : null;

    return view('supervisors.evaluations.index', compact(
        'students', 'evaluations', 'averages', 'attendanceMap', 'evaluationMap', 'selectedStudentId', 'selectedStudent'
    ));
}



public function create()
{
    $supervisorId = auth()->id();

    // Get assigned students
    $students = User::where('supervisor_id', $supervisorId)
                    ->where('role', 'student')
                    ->get();

    return view('evaluations.create', compact('students'));
}

public function store(Request $request)
{
    $request->validate([
        'student_id' => 'required|exists:users,id',
        'punctuality' => 'required|numeric|max:10',
        'attitude' => 'required|numeric|max:10',
        'skill_acquisition' => 'required|numeric|max:10',
        'report_quality' => 'required|numeric|max:10',
        'remarks' => 'nullable|string|max:255',
    ]);

    $studentId = $request->student_id;
    $supervisorId = auth()->id();

    $student = User::findOrFail($studentId);

    // Check for attendance
    $hasAttendance = Attendance::where('student_id', $studentId)->exists();
    if (!$hasAttendance) {
        return back()->with('error', 'Student has no attendance records to be evaluated!');
    }

    // Check for existing evaluation
    $alreadyEvaluated = Evaluation::where('student_id', $studentId)->exists();
    if ($alreadyEvaluated) {
        return back()->with('error', 'This student has already been evaluated.');
    }

    Evaluation::create([
        'student_id' => $studentId,
        'supervisor_id' => $supervisorId,
        'punctuality' => $request->punctuality,
        'attitude' => $request->attitude,
        'skill_acquisition' => $request->skill_acquisition,
        'report_quality' => $request->report_quality,
        'remarks' => $request->remarks,
    ]);

    return redirect()->route('evaluations.index')->with('success', 'Evaluation submitted successfully.');
}



}
