<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use App\Models\Attendance;
use App\Models\Evaluation;
use Illuminate\Support\Carbon;

class SupervisorController extends Controller
{

    public function updateLogbookEntry(Request $request, LogbookEntry $entry)
{
    $entry->lesson_learned = $request->lesson_learned;

    if ($request->hasFile('diagram')) {
        $path = $request->file('diagram')->store('logbook_diagrams', 'public');
        $entry->diagram_path = $path;
    }

    $entry->save();

    return back()->with('success', 'Logbook updated successfully!');
}


 // SupervisorController.php (viewLogbook method)

 public function viewLogbook(Request $request)
{
    $user = auth()->user();
    $selectedStudent = null;
    $logbookEntries = collect();
    $evaluation = null;
    $attendances = collect();

    // Determine student list based on user role
    if ($user->role == 'admin') {
        // Admin: See students who have supervisor and attendance or evaluation
        $students = User::where('role', 'student')
    ->whereNotNull('supervisor_id')
    ->where(function ($query) {
        $query->whereHas('attendances')
              ->orWhereHas('evaluations');
    })->get();

    } else {
        // Supervisor: Only their assigned students
        $students = User::where('supervisor_id', $user->id)
                        ->where('role', 'student')
                        ->get();
    }

    // Load selected student's data if selected
    if ($request->student_id) {
        $selectedStudent = User::find($request->student_id);

        $logbookEntries = Attendance::where('student_id', $request->student_id)
                                    ->orderBy('date', 'asc')
                                    ->get();

        $attendances = $logbookEntries; // Reuse same data

        // Admin sees any evaluation, supervisor sees only their own
        $evaluationQuery = Evaluation::where('student_id', $request->student_id);
        if ($user->role == 'supervisor') {
            $evaluationQuery->where('supervisor_id', $user->id);
        }

        $evaluation = $evaluationQuery->first();
    }

    return view('supervisors.view-logbook', compact(
        'students',
        'selectedStudent',
        'logbookEntries',
        'evaluation',
        'attendances'
    ));
}



public function uploadDiagram(Request $request, $studentId)
{
    $request->validate([
        'diagram' => 'nullable|image|max:2048'
    ]);

    $attendance = Attendance::where('student_id', $studentId)
                            ->where('supervisor_id', auth()->id())
                            ->where('date', now()->toDateString())
                            ->latest()
                            ->first();

    if ($attendance && $request->hasFile('diagram')) {
        $path = $request->file('diagram')->store('diagrams', 'public');
        $attendance->update(['diagram_path' => $path]);
    }

    return back()->with('success', 'Diagram uploaded successfully.');
}


    public function assignedStudents()
    {
        $students = User::where('supervisor_id', auth()->id())
                        ->where('role', 'student')
                        ->orderBy('created_at', 'desc')
                        ->get();

        $attendances = Attendance::where('supervisor_id', auth()->id())
                                ->whereDate('date', \Carbon\Carbon::now()->toDateString())
                                ->get();

        return view('supervisors.assigned-students', compact('students', 'attendances'));
    }



public function markAttendance(Request $request, $studentId)
{
    $request->validate([
        'lesson_learned' => 'nullable|string|max:500'
    ]);

    $supervisorId = auth()->id();
    $today = Carbon::now()->toDateString();

    // If no lesson_learned was sent (attendance button only)
    if (!$request->has('lesson_learned')) {
        Attendance::create([
            'student_id' => $studentId,
            'supervisor_id' => $supervisorId,
            'date' => $today,
            'lesson_learned' => 'No comments',
        ]);

        return back()->with('success', 'Attendance marked successfully.');
    }

    // If lesson_learned was submitted, update today's latest attendance
    $attendance = Attendance::where('student_id', $studentId)
        ->where('supervisor_id', $supervisorId)
        ->whereDate('date', $today)
        ->latest()
        ->first();

    if ($attendance) {
        $attendance->update([
            'lesson_learned' => $request->lesson_learned
        ]);

        return back()->with('success', 'Lesson learned updated successfully.');
    }

    return back()->with('success', 'Please mark attendance first before adding lesson learned.');
}



    // Show all supervisors
    public function index()
    {
        $supervisors = User::where('role', 'supervisor')->get();
        return view('admin.supervisors.index', compact('supervisors'));
    }

    // View students assigned to a supervisor
    public function showStudents($supervisorId)
    {
        $supervisor = User::with('students')->findOrFail($supervisorId);
        return view('admin.supervisors.students', compact('supervisor'));
    }
}
